<?php
require_once 'config.php';
require_once 'functions.php';

$content = file_get_contents("php://input");
$update = json_decode($content, true);

if (!$update) {
    exit;
}

if (isset($update['callback_query'])) {
    handle_callback_query($update['callback_query']);
    exit;
}

if (isset($update['message'])) {
    $message = $update['message'];
    $chat_id = $message['chat']['id'];
    $text = $message['text'] ?? '';
    $from_id = $message['from']['id'];
    
    if ($message['chat']['type'] === 'private') {
        save_user($chat_id);
    } elseif (in_array($message['chat']['type'], ['group', 'supergroup'])) {
        save_group($chat_id);
    }
    
    if (!get_bot_status() && $from_id != ADMIN_ID) {
        exit;
    }
    
    if ($from_id == ADMIN_ID) {
        $pending_action = get_pending_action($from_id);
        
        if ($pending_action === 'broadcast') {
            handle_broadcast($message);
            exit;
        } elseif ($pending_action === 'forward') {
            handle_forward($message);
            exit;
        } elseif (in_array($pending_action, ['block', 'unblock'])) {
            handle_block_unblock($message, $pending_action);
            exit;
        }
    }
    
    if (strpos($text, '/') === 0) {
        $command = explode(' ', $text)[0];
        
        switch ($command) {
            case '/start':
                send_welcome($chat_id);
                break;
                
            case '/admin':
                if ($from_id == ADMIN_ID) {
                    admin_panel($chat_id);
                }
                break;
                
            default:

                if (array_key_exists(substr($command, 1), $GLOBALS['MANUAL_ALIASES'])) {
                    handle_coin_request($message);
                } else {
                    sendMessage($chat_id, "دستور نامعتبر! از /start استفاده کنید.");
                }
        }
    } else {

        $text_lower = fa_num_to_en(strtolower(trim($text)));
        
        if (strpos($text_lower, 'طلا') !== false || strpos($text_lower, 'gold') !== false) {
            handle_gold_request($message);
        } else {
            handle_coin_request($message);
        }
    }
}

function send_welcome($chat_id) {
    $welcome_text = "👋 سلام! به ربات قیمت لحظه‌ای ارز دیجیتال خوش آمدید.\n"
                  . "💡 می‌توانید نام یا نماد ارز مورد نظر خود را ارسال کنید تا قیمت و اطلاعات آن را دریافت کنید.\n\n"
                  . "مثال‌ها:\n"
                  . "• بیت کوین\n"
                  . "• 0.5 اتریوم\n"
                  . "• 2 USDT";
    
    sendMessage($chat_id, $welcome_text);
}

function admin_panel($chat_id) {
    $keyboard = [
        'inline_keyboard' => [
    [['text' => '📊 آمار', 'callback_data' => 'stats']],
    [
        ['text' => '🔄 فوروارد', 'callback_data' => 'forward'],
        ['text' => '📢 همگانی', 'callback_data' => 'broadcast']
    ],
    [
        ['text' => '✅ آنبلاک', 'callback_data' => 'unblock'],
        ['text' => '🚫 بلاک', 'callback_data' => 'block']
    ],
    [
        ['text' => '🔴 خاموش', 'callback_data' => 'disable'],
        ['text' => '🟢 روشن', 'callback_data' => 'enable']
    ],
    [['text' => '📁 لیست گروه‌ها', 'callback_data' => 'list_groups']],
    [
        ['text' => '📢 کانال', 'url' => 'https://t.me/ZetaB0t'],
        ['text' => '👤 توسعه‌دهنده', 'url' => 'https://t.me/SpiderM9n']
    ]
]
    ];
    
    sendMessage($chat_id, "پنل مدیریت ربات : \n ➖➖➖➖➖➖➖➖‏➖➖➖", null, json_encode($keyboard));
}

function handle_broadcast($message) {
    if (isset($message['text'])) {
        $success = broadcast_to_users($message['text']);
        sendMessage($message['chat']['id'], "✅ پیام همگانی به $success کاربر ارسال شد.");
    } else {
        try {
            $forwarded = forward_to_all($message['chat']['id'], $message['message_id']);
            sendMessage($message['chat']['id'], "✅ پیام به $forwarded کاربر/گروه فوروارد شد.");
        } catch (Exception $e) {
            sendMessage($message['chat']['id'], "⚠️ خطا در فوروارد پیام: " . $e->getMessage());
        }
    }
    
    clear_pending_action($message['from']['id']);
}

function handle_forward($message) {
    try {
        $forwarded = forward_to_all($message['chat']['id'], $message['message_id']);
        sendMessage($message['chat']['id'], "✅ پیام به $forwarded کاربر/گروه فوروارد شد.");
    } catch (Exception $e) {
        sendMessage($message['chat']['id'], "⚠️ خطا در فوروارد پیام: " . $e->getMessage());
    }
    
    clear_pending_action($message['from']['id']);
}

function handle_block_unblock($message, $action) {
    $target_id = get_target_user_id($message);
    if (!$target_id) {
        sendMessage($message['chat']['id'], "⚠️ نتوانستم آیدی کاربر را استخراج کنم. لطفا آیدی عددی را بفرستید یا به پیام کاربر ریپلای کنید.");
        return;
    }
    
    if ($action == 'block') {
        add_blocked($target_id);
        sendMessage($message['chat']['id'], "🚫 کاربر با آیدی $target_id بلاک شد.");
    } else {
        remove_blocked($target_id);
        sendMessage($message['chat']['id'], "✅ کاربر با آیدی $target_id آنبلاک شد.");
    }
    
    clear_pending_action($message['from']['id']);
}

function handle_gold_request($message) {
    $text = fa_num_to_en(strtolower(trim($message['text'])));
    $chat_id = $message['chat']['id'];
    
    $amount = 1;
    if (preg_match('/^(\d+\.?\d*)\s*(?:گرم)?\s*(?:طلا|gold)/', $text, $matches)) {
        $amount = floatval($matches[1]);
    }
    
    $gold_price = get_gold_price();
    if ($gold_price === null) {
        sendMessage($chat_id, "⚠️ متاسفانه نتوانستم قیمت طلا را دریافت کنم. لطفا بعدا تلاش کنید.");
        return;
    }
    
    $gold_price_toman = $gold_price / 10;
    $total_price = $amount * $gold_price_toman;
    $date_time = date('Y/m/d | H:i:s');
    
    $msg = "💰 قیمت لحظه‌ای طلای ۱۸ عیار برای $amount گرم:\n"
         . number_format($total_price) . " تومان\n\n"
         . "📅 آخرین بروزرسانی: $date_time";
    
    $keyboard = [
        'inline_keyboard' => [
            [['text' => '➕ اضافه کردن به گروه', 'url' => 'https://t.me/' . BOT_USERNAME . '?startgroup=start']]
        ]
    ];
    
    sendMessage($chat_id, $msg, null, json_encode($keyboard));
}

function handle_coin_request($message) {
    global $MANUAL_ALIASES;
    
    $text = fa_num_to_en(strtolower(trim($message['text'])));
    $chat_id = $message['chat']['id'];
    
    if (strpos($text, '/') === 0) {
        $text = substr($text, 1);
    }
    
    $amount = 1;
    $coin_key = $text;
    
    $parts = explode(' ', $text);
    if (count($parts) > 1 && is_numeric(str_replace('.', '', $parts[0]))) {
        $amount = floatval($parts[0]);
        $coin_key = implode(' ', array_slice($parts, 1));
    } elseif (preg_match('/^(\d+\.?\d*)\s*([^\d]+)$/', $text, $matches)) {
        $amount = floatval($matches[1]);
        $coin_key = trim($matches[2]);
    }
    
    $coin_key = $MANUAL_ALIASES[$coin_key] ?? $coin_key;
    
    $coins = get_coin_data();
    if (!$coins) {
        sendMessage($chat_id, "⚠️ مشکلی در دریافت داده از API پیش آمد. لطفا بعدا تلاش کنید.");
        return;
    }
    
    $coin = $coins[$coin_key] ?? null;
    if (!$coin) {
        sendMessage($chat_id, "⚠️ ارز '$text' پیدا نشد. لطفا نام دقیق ارز یا عدد + نام ارز رو بفرستید.");
        return;
    }
    
    $usd_price = floatval($coin['usdt']);
    $toman_price = floatval($coin['irr']);
    $change_percent = floatval($coin['dayChange']);
    
    $profit_usd = $amount * $usd_price * ($change_percent / 100);
    $profit_toman = $amount * $toman_price * ($change_percent / 100);
    $date_time = date('Y/m/d | H:i:s');
    
    $symbol = null;
    foreach ($coins as $sym => $c) {
        if ($c === $coin && strlen($sym) <= 5) {
            $symbol = strtoupper($sym);
            break;
        }
    }
    
    if (!$symbol) {
        $symbol = strtoupper($coin_key);
    }
    
    $amount_str = format_number($amount);
    $usd_total_str = format_number($amount * $usd_price);
    $toman_total_str = number_format($amount * $toman_price);
    $profit_usd_str = format_number($profit_usd, 6);
    $profit_toman_str = number_format($profit_toman);
    
    $msg = "💰 $amount_str $symbol = $$usd_total_str\n"
         . "💶 $toman_total_str تومان\n\n"
         . ($change_percent >= 0 ? '🟢' : '🔴') . " درصد تغییرات: " . number_format($change_percent, 2) . "%\n"
         . ($change_percent >= 0 ? '🔺 مقدار سود' : '🔻 مقدار ضرر') . ": "
         . "$profit_usd_str $ / $profit_toman_str تومان\n\n"
         . "📅 $date_time";
    
    $keyboard = [
        'inline_keyboard' => [
            [['text' => '➕ اضافه کردن به گروه', 'url' => 'https://t.me/' . BOT_USERNAME . '?startgroup=start']]
        ]
    ];
    
    sendMessage($chat_id, $msg, null, json_encode($keyboard));
}

function handle_callback_query($callback_query) {
    global $MANUAL_ALIASES;
    
    $user_id = $callback_query['from']['id'];
    $chat_id = $callback_query['message']['chat']['id'];
    $data = $callback_query['data'];
    
    if ($user_id != ADMIN_ID) {
        sendMessage($chat_id, "شما دسترسی ندارید!");
        return;
    }
    
    if ($data == 'stats') {
        $users_count = 0;
        if (file_exists(USERS_FILE)) {
            $users = file(USERS_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $users_count = count($users);
        }
        
        $groups_count = 0;
        if (file_exists(GROUPS_FILE)) {
            $groups = file(GROUPS_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $groups_count = count($groups);
        }
        
        $blocked_count = 0;
        if (file_exists(BLOCKED_FILE)) {
            $blocked = file(BLOCKED_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            $blocked_count = count($blocked);
        }
        
        $status = get_bot_status() ? '🟢 روشن' : '🔴 خاموش';
        
        $msg = "📊 آمار ربات:\n\n"
             . "👥 تعداد کاربران: $users_count\n"
             . "📁 تعداد گروه‌ها: $groups_count\n"
             . "🚫 کاربران بلاک شده: $blocked_count\n"
             . "🔴 وضعیت ربات: $status";
        
        sendMessage($chat_id, $msg);
    } elseif ($data == 'list_groups') {
        if (!file_exists(GROUPS_FILE)) {
            sendMessage($chat_id, "📁 هیچ گروهی ثبت نشده.");
            return;
        }
        
        $groups = file(GROUPS_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        if (empty($groups)) {
            sendMessage($chat_id, "📁 هیچ گروهی ثبت نشده.");
            return;
        }
        
        $msg = "📁 لیست گروه‌ها:\n\n";
        $count = 0;
        
        foreach ($groups as $group_id) {
            if ($count >= 50) break;
            
            $msg .= "- آیدی: $group_id\n";
            $count++;
        }
        
        if (count($groups) > 50) {
            $msg .= "\nو " . (count($groups) - 50) . " گروه دیگر...";
        }
        
        sendMessage($chat_id, $msg);
    } elseif (in_array($data, ['enable', 'disable'])) {
        set_bot_status($data == 'enable');
        $status_msg = $data == 'enable' ? "🟢 ربات روشن شد." : "🔴 ربات خاموش شد. (فقط ادمین می‌تواند با ربات کار کند)";
        sendMessage($chat_id, $status_msg);
    } elseif (in_array($data, ['broadcast', 'forward', 'block', 'unblock'])) {
        set_pending_action($user_id, $data);
        
        if ($data == 'broadcast') {
            $msg = "📢 لطفا متن یا پیام مورد نظر برای ارسال همگانی را ارسال یا فوروارد کنید.";
        } elseif ($data == 'forward') {
            $msg = "📨 لطفا پیام مورد نظر را فوروارد کنید یا پیام را ارسال/ریپلای کنید تا به همهٔ کاربران و گروه‌ها فوروارد شود.";
        } elseif ($data == 'block') {
            $msg = "🚫 لطفا آیدی کاربر را ارسال کنید یا به پیام او ریپلای کنید تا بلاک شود.";
        } else {
            $msg = "✅ لطفا آیدی کاربر را ارسال کنید یا به پیام او ریپلای کنید تا آنبلاک شود.";
        }
        
        sendMessage($chat_id, $msg);
    }
    
    $answer_url = API_URL . "answerCallbackQuery";
    $answer_data = ['callback_query_id' => $callback_query['id']];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $answer_url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $answer_data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_exec($ch);
    curl_close($ch);
}
?>